#include <TeDatabase.h>
#include <TePostgreSQL.h>
#include <TeInitRasterDecoders.h>
#include <TeImportRaster.h>

#include <HidroBlocks.h>
#include <HidroTestProgress.h>
#include <HidroTestUtils.h>

bool createDatabase( TeRaster* demRaster, TeRaster* soilsRaster, TeRaster* vegetationRaster );
bool createRasterLayer( std::string layerName, TeRaster* raster, TeDatabase* db );
bool openRasterTiff( std::string rasterName, TeRaster* &raster );
bool saveRasterLayer( std::string layerName, TeRaster* raster );
bool openRasterDB( std::string rasterName, TeRaster* &raster );

int main()
{
  // set the progress interfae
  HidroTestProgress* progressSTio = new HidroTestProgress();
  TeProgress::setProgressInterf( progressSTio );

	// Initialize the raster decoder tool
	TeInitRasterDecoders();

  // Open the DEM tif
  TeRaster* demRaster = 0;
  if( !openRasterDB( "DEM", demRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }
  /*if( !openRasterTiff( "..\\..\\data\\blocks\\dem.tif", demRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }*/

  // Open the soils tif
  TeRaster* soilsRaster = 0;
  if( !openRasterDB( "soils", soilsRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }
  /* if( !openRasterTiff( "..\\..\\data\\blocks\\soils.tif", soilsRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }/**/

  // Open the vegetation
  TeRaster* vegetationRaster = 0; 
  if( !openRasterDB( "vegetation", vegetationRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }
  /*if( !openRasterTiff( "..\\..\\data\\blocks\\vegetation.tif", soilsRaster ) )
  {
    cout << "Press Enter\n";
		getchar();
    return false;
  }/**/

  // -----------------   Create Database (optional) -------------------
  /*if( !createDatabase(demRaster, soilsRaster, vegetationRaster) != 0 )
  {
    cout << "Fail to create the adatabase." << endl << endl;
		cout << "Press Enter\n";
		getchar();
    return 1;
  }/**/

  // The output blocks raster
  TeRaster* blocksRaster = 0;

  HidroBlocks blocksGenerator( demRaster,
    soilsRaster,
    vegetationRaster,
    "..\\..\\data\\blocks\\cell.hig",
    blocksRaster,
    "..\\..\\data\\blocks\\blocks.hig",
    "..\\..\\data\\blocks\\grupos.hig" );

  if( !blocksGenerator.execute() )
  {
    cout << "fail in blocksGenerator.\n\n";
    getchar();
    return false;
  }

  // save the blocksRaster in database
  saveRasterLayer( "blocks", blocksRaster );

  // delete the rasters
  if( demRaster != 0 )
  {
    delete demRaster;
  }
  if( soilsRaster != 0 )
  {
    delete soilsRaster;
  }
  if( vegetationRaster != 0 )
  {
    delete vegetationRaster;
  }
  if( blocksRaster != 0 )
  {
    delete blocksRaster;
  }

	cout << "\nPress enter...";
	getchar();
	return true;
}


bool
createDatabase(TeRaster* demRaster, TeRaster* soilsRaster, TeRaster* vegetationRaster)
{
  // Database server parameters
  string host = "localhost";
  string dbname = "BlocksTest";
  string user = "postgres";
  string password = "postgres";

  // Connect to the database
  TeDatabase* db = new TePostgreSQL();
  if( !db->newDatabase( dbname, user, password,host ) )
  {
    cout << "Error: " << db->errorMessage() << endl << endl;    
    return 1;
  }

  // Import rasters to database
  if( !createRasterLayer( "DEM", demRaster, db ) )
  {
    db->close();
		cout << "Fail to import the DEM image\n\n!";		
		return false;
  }
  if( !createRasterLayer( "soils", soilsRaster, db ) )
  {
    db->close();
		cout << "Fail to import the soils image\n\n!";		
		return false;
  }
  if( !createRasterLayer( "vegetation", vegetationRaster, db ) )
  {
    db->close();
		cout << "Fail to import the vegetation image\n\n!";		
		return false;
  } 

  // Close the database
	db->close();

  return 0;
}

bool
createRasterLayer( std::string layerName, TeRaster* raster, TeDatabase* db )
{
	// Check whether there is a layer with this name in the database
	if( db->layerExist( layerName ) )
	{		
		cout << "The database already has an infolayer with the name \"";
		cout << layerName << "\"!" << endl << endl;		
		return false;
	}

  // Create the layer
	TeLayer* layer = new TeLayer( layerName, db, raster->projection() );
	if (layer->id() <= 0)
	{		
		cout << "The destination layer \"" << layerName << "\" could not be created!\n" << db->errorMessage() << endl;		
		return false;
	}

	// Import the DEM image to the layer
  if( !TeImportRaster( layer, raster, 512, 512, TeRasterParams::TeNoCompression, "", -9999, false, TeRasterParams::TeNoExpansible ) )
	{		
		cout << "Fail to import the raster to layer\"" << layerName << "\".\n\n!";		
		return false;
	}

  return true;
}

bool openRasterTiff( std::string rasterName, TeRaster* &raster )
{
  raster = new TeRaster( rasterName );
  if( !raster->init() )
	{
		cout << "Cannot access the \"" << rasterName << "\" input tif!" << endl << endl;		
		return false;
	}

  return true;
}

bool saveRasterLayer( std::string layerName, TeRaster* raster )
{
  // Database server parameters
  string host = "localhost";
  string dbname = "BlocksTest";
  string user = "postgres";
  string password = "postgres";

  // Connect to the database
  TeDatabase* db = new TePostgreSQL();
  if( !db->connect(host, user, password, dbname ) )
  {
    cout << "Error: " << db->errorMessage() << endl << endl;    
    return false;
  }

  // Import rasters to database
  if( !createRasterLayer( layerName, raster, db ) )
  {
    db->close();
		cout << "Fail to import the \"" << layerName << "\"image\n\n!";		
		return false;
  }

  // Close the database
	db->close();

  return 0;
}

bool openRasterDB( std::string rasterName, TeRaster* &raster )
{
  // Database server parameters
  string host = "localhost";
  string dbname = "BlocksTest";
  string user = "postgres";
  string password = "postgres";

  // Connect to the database
  TeDatabase* db = new TePostgreSQL();
  if( !db->connect( host, user, password, dbname ) )
  {
    cout << "Error: " << db->errorMessage() << endl << endl;    
    return false;
  }

  HidroTestUtils utils( db );

  TeLayer* layer = NULL;
	layer = utils.getLayerByName( rasterName );
  raster = NULL;
  raster = layer->raster();

  return true;
}